/* -*- mode: C++; tab-width: 4; c-basic-offset: 4; -*- */

/* AbiSource
 * 
 * Copyright (C) 2007 Philippe Milot <PhilMilot@gmail.com>
 * 
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  
 * 02111-1307, USA.
 */


// Class definition include
#include "ie_imp_OpenXML.h"

// Internal includes
#include <OXML_Types.h>
#include <OXML_Element.h>
#include <OXML_Section.h>
#include <OXML_Document.h>
#include <OXML_Style.h>
#include <OXML_Theme.h>
#include <OXMLi_PackageManager.h>

// AbiWord includes
#include <ut_types.h>
#include <ut_assert.h>

// External includes
#include <iostream>
#include <gsf/gsf-infile.h>
#include <gsf/gsf-infile-zip.h>


/**
 * Constructor
 */
IE_Imp_OpenXML::IE_Imp_OpenXML (PD_Document * pDocument)
  : IE_Imp (pDocument)
{
}


/*
 * Destructor
 */
IE_Imp_OpenXML::~IE_Imp_OpenXML ()
{
	_cleanup();
}


//////// UNIT TESTING FUNCTIONS (will be moved to another file eventually) //////////////
void testElement()
{
	UT_DEBUGMSG(("Running unit test for OXML_Element...\n"));
	OXML_SharedElement elem(new OXML_Element("elementId", P_TAG, BLOCK));
	UT_DEBUGMSG(("Retrieving basic data\n"));
	UT_ASSERT(elem->getId().compare("elementId") == 0);
	UT_ASSERT(elem->getTag() == P_TAG);
	UT_ASSERT(elem->getType() == BLOCK);

	UT_DEBUGMSG(("Running children tests\n"));
	UT_ASSERT(elem->getElement("childId").get() == NULL);
	OXML_SharedElement child(new OXML_Element("childId", R_TAG, SPAN));
	UT_ASSERT(elem->appendElement( OXML_SharedElement() ) == UT_ERROR);
	UT_ASSERT(elem->appendElement(child) == UT_OK);
	UT_ASSERT(elem->getElement("childId")->getId().compare(child->getId()) == 0);

	UT_DEBUGMSG(("Cleaning data\n"));
	UT_ASSERT(elem->clearChildren() == UT_OK);
	UT_ASSERT(elem->getElement("childId").get() == NULL);
	
	UT_DEBUGMSG(("Unit test successful.\n"));
}

void testSection()
{
	UT_DEBUGMSG(("Running unit test for OXML_Section...\n"));
	OXML_SharedSection sect(new OXML_Section("sectionId"));
	UT_ASSERT(sect->getId().compare("sectionId") == 0);


	UT_DEBUGMSG(("Setting header\n"));
	sect->setHeaderId("normalHdr", NORMAL);
	UT_DEBUGMSG(("Setting footer\n"));
	sect->setFooterId("oddpageFtr", ODD_PAGE);
	UT_DEBUGMSG(("Getting existing header\n"));
	UT_ASSERT(sect->getHeaderId(NORMAL).compare("normalHdr") == 0);
	UT_DEBUGMSG(("Getting existing footer\n"));
	UT_ASSERT(sect->getFooterId(ODD_PAGE).compare("oddpageFtr") == 0);
	UT_DEBUGMSG(("Getting non-existing header\n"));
	UT_ASSERT(sect->getHeaderId(ODD_PAGE).compare("") == 0);
	UT_DEBUGMSG(("Getting non-existing footer\n"));
	UT_ASSERT(sect->getFooterId(NORMAL).compare("") == 0);

	UT_ASSERT(sect->getElement("childId").get() == NULL);
	OXML_SharedElement child(new OXML_Element("childId", P_TAG, BLOCK));
	UT_ASSERT(sect->appendElement(child) == UT_OK);
	UT_ASSERT(sect->appendElement( OXML_SharedElement() ) == UT_ERROR);
	UT_ASSERT(sect->getElement("childId")->getId().compare(child->getId()) == 0);

	UT_ASSERT(sect->clearChildren() == UT_OK);
	UT_ASSERT(sect->getElement("childId").get() == NULL);

	UT_DEBUGMSG(("Unit test successful.\n"));
}

void testDocument()
{
	UT_DEBUGMSG(("Running unit test for OXML_Document...\n"));
	UT_DEBUGMSG(("Retrieving the instance\n"));
	OXML_Document* doc = OXML_Document::getInstance();
	UT_ASSERT(doc != NULL);

	UT_DEBUGMSG(("Getting non-existing styles\n"));
	UT_ASSERT(NULL == doc->getStyleById("id1").get());
	UT_ASSERT(NULL == doc->getStyleByName("style1").get());

	UT_DEBUGMSG(("Adding new styles\n"));
	OXML_SharedStyle style1(new OXML_Style("id1", "style1"));
	UT_ASSERT(UT_OK == doc->addStyle(style1));
	UT_ASSERT(UT_ERROR == doc->addStyle( OXML_SharedStyle() ));
	UT_ASSERT(UT_OK == doc->addStyle("id2", "style2", NULL));

	UT_DEBUGMSG(("Getting existing styles\n"));
	UT_ASSERT(0 == doc->getStyleById("id1")->getId().compare("id1"));
	UT_ASSERT(0 == doc->getStyleByName("style1")->getName().compare("style1"));
	UT_ASSERT(0 == doc->getStyleById("id2")->getId().compare("id2"));
	UT_ASSERT(0 == doc->getStyleByName("style2")->getName().compare("style2"));

	UT_DEBUGMSG(("Clearing styles\n"));
	UT_ASSERT(UT_OK == doc->clearStyles());

	UT_DEBUGMSG(("Getting non-existing styles\n"));
	UT_ASSERT(NULL == doc->getStyleById("id1").get());
	UT_ASSERT(NULL == doc->getStyleByName("style1").get());
	UT_ASSERT(NULL == doc->getStyleById("id2").get());
	UT_ASSERT(NULL == doc->getStyleByName("style2").get());


	OXML_SharedSection hdr(new OXML_Section("headerId"));
	OXML_SharedSection ftr(new OXML_Section("footerId"));

	//TODO: test the headers and footers

	UT_DEBUGMSG(("Getting non-existing section\n"));
	UT_ASSERT(doc->getSection("childId").get() == NULL);

	UT_DEBUGMSG(("Appending section\n"));
	OXML_SharedSection child(new OXML_Section("childId"));
	UT_ASSERT(doc->appendSection(child) == UT_OK);
	UT_ASSERT(UT_ERROR == doc->appendSection( OXML_SharedSection() ));

	UT_DEBUGMSG(("Getting existing section\n"));
	UT_ASSERT(doc->getSection("childId")->getId().compare(child->getId()) == 0);

	UT_DEBUGMSG(("Clearing children\n"));
	UT_ASSERT(doc->clearSections() == UT_OK);
	UT_ASSERT(doc->getSection("childId").get() == NULL);

	UT_DEBUGMSG(("Destroying instance\n"));
	OXML_Document::destroyInstance();

	UT_DEBUGMSG(("Unit test successful.\n"));
}

void testStyle()
{
	UT_DEBUGMSG(("Running unit test for OXML_Style...\n"));
	OXML_SharedStyle style(new OXML_Style("styleId", "styleName"));
	UT_ASSERT(style->getId().compare("styleId") == 0);
	UT_ASSERT(style->getName().compare("styleName") == 0);
	UT_DEBUGMSG(("Unit test successful.\n"));
}

void testTheme()
{
	UT_DEBUGMSG(("Running unit test for OXML_Theme...\n"));
	OXML_SharedTheme theme(new OXML_Theme());

	UT_DEBUGMSG(("Testing color scheme\n"));
	UT_ASSERT(0 == theme->getColor(LIGHT1).compare(""));
	theme->setColor(LIGHT1, "#0FFFFF");
	UT_ASSERT(0 == theme->getColor(LIGHT1).compare("#0FFFFF"));
	UT_ASSERT(0 == theme->getColor(FOLLOWED_HYPERLINK).compare(""));

	UT_DEBUGMSG(("Testing major font scheme\n"));
	UT_ASSERT(0 == theme->getMajorFont("cs").compare(""));
	theme->setMajorFont("cs", "Arial");
	UT_ASSERT(0 == theme->getMajorFont("cs").compare("Arial"));

	UT_DEBUGMSG(("Testing minor font scheme\n"));
	UT_ASSERT(0 == theme->getMinorFont("cs").compare(""));
	theme->setMinorFont("cs", "Arial");
	UT_ASSERT(0 == theme->getMinorFont("cs").compare("Arial"));

}

//////// END OF UNIT TESTING FUNCTIONS //////////////

/**
 * Import the given file
 */
UT_Error IE_Imp_OpenXML::_loadFile (GsfInput * oo_src)
{
	UT_DEBUGMSG(("\n\n\nLoading an OpenXML file\n"));

	UT_Error ret = UT_OK;

	GsfInfile * pGsfInfile = GSF_INFILE (gsf_infile_zip_new (oo_src, NULL));
    
	if (pGsfInfile == NULL) {
		return UT_ERROR;
	}

	OXMLi_PackageManager * mgr = OXMLi_PackageManager::getNewInstance();
	if (mgr == NULL) {
		g_object_unref (G_OBJECT(pGsfInfile));
		_cleanup();
		return UT_ERROR;
	}

	mgr->setContainer(pGsfInfile);

	UT_DEBUGMSG(("Building the data model...\n"));
	//These calls build the data model
	if (UT_OK != (ret = mgr->parseDocumentTheme()))
	{
		_cleanup();
		return ret;
	}

	if (UT_OK != (ret = mgr->parseDocumentSettings()))
	{
		_cleanup();
		return ret;
	}

	if (UT_OK != (ret = mgr->parseDocumentStyles()))
	{
		_cleanup();
		return ret;
	}

	if (UT_OK != (ret = mgr->parseDocumentStream()))
	{
		_cleanup();
		return ret;
	}

	UT_DEBUGMSG(("Data model built.  Building piecetable...\n"));

	OXML_Document * doc = OXML_Document::getInstance();
	if (doc == NULL) {
		_cleanup();
		return UT_ERROR;
	}

	if (UT_OK != (ret = doc->addToPT( getDoc() ))) //This builds the piecetable from the data model
	{
		_cleanup();
		return ret;
	}

	_cleanup();

	UT_DEBUGMSG(("Finished loading OpenXML file\n\n\n\n"));

	return ret;

}

void IE_Imp_OpenXML::_cleanup ()
{
	OXMLi_PackageManager::destroyInstance();
	OXML_Document::destroyInstance();
}

