/* -*- mode: C++; tab-width: 4; c-basic-offset: 4; -*- */

/* AbiCollab- Code to enable the modification of remote documents.
 * Copyright (C) 2005 by Martin Sevior
 * Copyright (C) 2006 by Marc Maurer <uwog@uwog.net>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA
 * 02111-1307, USA.
 */

#ifdef ABI_PLUGIN_BUILTIN
#define abi_plugin_register abipgn_abicollab_register
#define abi_plugin_unregister abipgn_abicollab_unregister
#define abi_plugin_supports_version abipgn_abicollab_supports_version
#endif

#include "xap_Module.h"
#include "AbiCollab_Plugin.h"
#include "xap_App.h"
#include "xap_Frame.h"
// needed for menu's
#include "ap_Menu_Id.h"
#include "ev_Menu_Actions.h"
#include "ev_Menu.h"
#include "ev_Menu_Layouts.h"
#include "ev_Menu_Labels.h"
#include "ev_EditMethod.h"
#include "xap_Menu_Layouts.h"
#include "AbiCollabSessionManager.h"
#include <xp/ap_Dialog_CollaborationJoin.h>
#include <xp/ap_Dialog_CollaborationAccounts.h>
#include "fv_View.h"
#include "pd_Document.h"
#include "ut_vector.h"
#include "xap_DialogFactory.h"

// forward declarations
void s_abicollab_add_menus();
void s_abicollab_remove_menus();

// -----------------------------------------------------------------------
//
//      Abiword Plugin Interface 
//
// -----------------------------------------------------------------------

ABI_PLUGIN_DECLARE(AbiCollab)
  
ABI_FAR_CALL
int abi_plugin_register (XAP_ModuleInfo * mi)
{
	mi->name = "AbiWord Collaboration";
	mi->desc = "This plugin allows real-time collaborative document editting";
	mi->version = ABI_VERSION_STRING;
	mi->author = "Martin Sevior <msevior@physics.unimelb.edu.au>\nMarc Maurer <uwog@uwog.net>";
	mi->usage = "";
	
	s_abicollab_add_menus();
	
	// On Windows, we must share our HMODULE/HINSTANCE so we can do gui
	#ifdef WIN32
	AbiCollabSessionManager::getManager()->setInstance((HINSTANCE)s_hModule);
	#endif
	
	// register all available account handlers
	AbiCollabSessionManager::getManager()->registerAccountHandlers();

	// finally, register all dialogs we'll use as well    
   	AbiCollabSessionManager::getManager()->registerDialogs();
   	
   	// load the settings from the profile
   	AbiCollabSessionManager::getManager()->loadProfile();
	
	
   		
	return 1;
}


ABI_FAR_CALL
int abi_plugin_unregister (XAP_ModuleInfo * mi)
{
	mi->name = 0;
	mi->desc = 0;
	mi->version = 0;
	mi->author = 0;
	mi->usage = 0;
	
	s_abicollab_remove_menus();
	
	// kill all active sessions
	AbiCollabSessionManager::getManager()->disconnectSessions();
	
	// store the settings to the profile
	AbiCollabSessionManager::getManager()->storeProfile();

	// register all available account handlers
	AbiCollabSessionManager::getManager()->unregisterAccountHandlers();

	// unregister all dialogs we use
	AbiCollabSessionManager::getManager()->unregisterDialogs();

	return 1;
}


ABI_FAR_CALL
int abi_plugin_supports_version (UT_uint32 major, UT_uint32 minor, UT_uint32 release)
{
	return 1; 
}


// -----------------------------------------------------------------------
//
//      Abiword Plugin Menu Structure
//
// -----------------------------------------------------------------------

// FIXME: make these translatable strings
static const char * szCollaboration = "&Collaborate";
static const char * szCollaborationTip = "Collaborate over the internet or local network";

static const char * szCollaborationOffer = "Share Document";
static const char * szCollaborationOfferTip = "Offer the current document for collaboration";

static const char * szCollaborationJoin = "Available Documents";
static const char * szCollaborationJoinTip = "Open a Document for collaboration";

static const char * szCollaborationAccounts = "Accounts";
static const char * szCollaborationAccountsTip = "Manage collaboration accounts";

static const char * szEndCollaboration = "EndCollaboration";


// some function prototypes
bool s_abicollab_offer(AV_View* v, EV_EditMethodCallData *d);
bool s_abicollab_join(AV_View* v, EV_EditMethodCallData *d);
bool s_abicollab_accounts(AV_View* v, EV_EditMethodCallData *d);

#define ABIWORD_VIEW  	FV_View * pView = static_cast<FV_View *>(pAV_View)

/*!
 * returns checked true if current document is part of a collaboration
 */
Defun_EV_GetMenuItemState_Fn(collab_GetState_Joined)
{
	AbiCollabSessionManager* pManager = AbiCollabSessionManager::getManager();
 	const UT_GenericVector<AccountHandler *> vecAccounts = pManager->getAccounts();

	bool atLeastOneOnline = false;
	for (UT_sint32 i = 0; i < vecAccounts.getItemCount(); i++)
	{
		AccountHandler* pHandler = vecAccounts.getNthItem(i);
		if (pHandler && pHandler->isOnline())
		{
			atLeastOneOnline = true;
			break;
		}
	}
	if (!atLeastOneOnline)
		return EV_MIS_Gray;
	
	ABIWORD_VIEW;
	UT_return_val_if_fail (pView, EV_MIS_Gray);
	PD_Document* pDoc = pView->getDocument();
	UT_return_val_if_fail (pDoc, EV_MIS_Gray);
	
	if (pManager->isInSession(pDoc))
	{
		if (pManager->isLocallyControlled(pDoc))
			return EV_MIS_Toggled;
		else
			return EV_MIS_Gray;
	}
	return EV_MIS_ZERO;
}

/*!
 * returns grayed if there are no active connections
 */
Defun_EV_GetMenuItemState_Fn(collab_GetState_AnyActive)
{
	AbiCollabSessionManager* pManager = AbiCollabSessionManager::getManager();
 	const UT_GenericVector<AccountHandler*> vecAccounts = pManager->getAccounts();

	for (UT_sint32 i = 0; i < vecAccounts.getItemCount(); i++)
	{
		AccountHandler* pHandler = vecAccounts.getNthItem(i);
		if (pHandler && pHandler->isOnline())
			return EV_MIS_ZERO;
	}

	return EV_MIS_Gray;
}



/*!
 * This implements the "Collaborate" menu item's placed under "file"
 */
void s_abicollab_add_menus()
{
    // First we need to get a pointer to the application itself.
    XAP_App *pApp = XAP_App::getApp();
    EV_EditMethodContainer* pEMC = pApp->getEditMethodContainer();
    int frameCount = pApp->getFrameCount();
    XAP_Menu_Factory * pFact = pApp->getMenuFactory();    
    EV_Menu_ActionSet* pActionSet = pApp->getMenuActionSet();
	
	// TODO: make this a translatable set of strings
	// const XAP_StringSet * pSS = pApp->getStringSet();
    
	// The Collaboration menu item
	XAP_Menu_Id collabId = pFact->addNewMenuBefore("Main", NULL, AP_MENU_ID_HELP, EV_MLF_BeginSubMenu);
    pFact->addNewLabel(NULL, collabId, szCollaboration, szCollaborationTip);
    EV_Menu_Action* myCollaborationAction = new EV_Menu_Action (
		collabId,    			 // id that the layout said we could use
		1,                      // yes, we have a sub menu.
		0,                      // no, we don't raise a dialog.
		0,                      // no, we don't have a checkbox.
		0,                      // no radio buttons for me, thank you
		NULL,                   // no callback function to call.
		NULL,                   // Function for whether not label is enabled/disabled checked/unchecked
		NULL                    // Function to compute Menu Label "Dynamic Label"
	);
	pActionSet->addAction(myCollaborationAction);

	// The Start Collaboration connect item
	XAP_Menu_Id collabOfferId = pFact->addNewMenuAfter("Main", NULL, collabId, EV_MLF_Normal);
    pFact->addNewLabel(NULL, collabOfferId, szCollaborationOffer, szCollaborationOfferTip);
	EV_Menu_Action* myActionOffer = new EV_Menu_Action (
		collabOfferId,   	  // id that the layout said we could use
		0,                      // no, we don't have a sub menu.
		0,                      // yes, we raise a dialog.
		1,                      // yes, we have a checkbox.
		0,                      // no radio buttons for me, thank you
		"s_abicollab_offer",    // name of callback function to call.
		collab_GetState_Joined, // Function for whether not label is enabled/disabled checked/unchecked
		NULL                    // Function to compute Menu Label "Dynamic Label"
	);
	pActionSet->addAction(myActionOffer);
	EV_EditMethod *myEditMethodOffer = new EV_EditMethod (
		"s_abicollab_offer",    // name of callback function
		s_abicollab_offer,      // callback function itself.
		0,                      // no additional data required.
		""                      // description -- allegedly never used for anything
	);
	pEMC->addEditMethod(myEditMethodOffer);

	// The Join Collaboration connect item
	XAP_Menu_Id collabJoinId = pFact->addNewMenuAfter("Main", NULL, collabOfferId, EV_MLF_Normal);
    pFact->addNewLabel(NULL, collabJoinId, szCollaborationJoin, szCollaborationJoinTip);
	EV_Menu_Action* myActionJoin = new EV_Menu_Action (
		collabJoinId,   		// id that the layout said we could use
		0,                      // no, we don't have a sub menu.
		1,                      // yes, we raise a dialog.
		0,                      // no, we don't have a checkbox.
		0,                      // no radio buttons for me, thank you
		"s_abicollab_join",     // name of callback function to call.
		collab_GetState_AnyActive,  // Function for whether not label is enabled/disabled checked/unchecked
		NULL                    // Function to compute Menu Label "Dynamic Label"
	);
	pActionSet->addAction(myActionJoin);
	EV_EditMethod *myEditMethodJoin = new EV_EditMethod (
		"s_abicollab_join",     // name of callback function
		s_abicollab_join,       // callback function itself.
		0,                      // no additional data required.
		""                      // description -- allegedly never used for anything
	);
	pEMC->addEditMethod(myEditMethodJoin);

	// The Join Collaboration connect item
	XAP_Menu_Id collabAccountsId = pFact->addNewMenuAfter("Main", NULL, collabJoinId, EV_MLF_Normal);
    pFact->addNewLabel(NULL, collabAccountsId, szCollaborationAccounts, szCollaborationAccountsTip);
	EV_Menu_Action* myActionAccounts = new EV_Menu_Action (
		collabAccountsId,   		// id that the layout said we could use
		0,                      // no, we don't have a sub menu.
		1,                      // yes, we raise a dialog.
		0,                      // no, we don't have a checkbox.
		0,                      // no radio buttons for me, thank you
		"s_abicollab_accounts",     // name of callback function to call.
		NULL,                   // Function for whether not label is enabled/disabled checked/unchecked
		NULL                    // Function to compute Menu Label "Dynamic Label"
	);
	pActionSet->addAction(myActionAccounts);
	EV_EditMethod *myEditMethodAccounts = new EV_EditMethod (
		"s_abicollab_accounts",     // name of callback function
		s_abicollab_accounts,       // callback function itself.
		0,                      // no additional data required.
		""                      // description -- allegedly never used for anything
	);
	pEMC->addEditMethod(myEditMethodAccounts);

	// End of the Collaboration menu
	XAP_Menu_Id endCollaborationId = pFact->addNewMenuAfter("Main", NULL, collabAccountsId, EV_MLF_EndSubMenu);
	pFact->addNewLabel(NULL, endCollaborationId, szEndCollaboration, NULL);
	EV_Menu_Action* myEndCollaborationAction = new EV_Menu_Action (
		endCollaborationId,     // id that the layout said we could use
		0,                      // no, we don't have a sub menu.
		0,                      // no, we raise a dialog.
		0,                      // no, we don't have a checkbox.
		0,                      // no radio buttons for me, thank you
		NULL,                   // name of callback function to call.
		NULL,                   // Function for whether not label is enabled/disabled checked/unchecked
		NULL                    // Function to compute Menu Label "Dynamic Label"
	);
    pActionSet->addAction(myEndCollaborationAction);
    

    // We need to go through and add the menu element to each "frame" 
    // of the application.  We can iterate through the frames by doing
    // XAP_App::getFrameCount() to tell us how many frames there are,
    // then calling XAP_App::getFrame(i) to get the i-th frame.
    for(int i = 0; i < frameCount;++i)
    {
        // Get the current frame that we're iterating through.
		XAP_Frame* pFrame = pApp->getFrame(i);
		pFrame->rebuildMenus();
    }
}

/*!
 * Remove the menu items unpon unloading the plugin.
 */
void s_abicollab_remove_menus()
{
	// First we need to get a pointer to the application itself.
	XAP_App *pApp = XAP_App::getApp();
	
	// remove the edit method
	EV_EditMethodContainer* pEMC = pApp->getEditMethodContainer();
	EV_EditMethod * pEM;

	pEM = ev_EditMethod_lookup ( "s_abicollab_offer" ) ;
	pEMC->removeEditMethod ( pEM ) ;
	DELETEP( pEM ) ;

	pEM = ev_EditMethod_lookup ( "s_abicollab_join" ) ;
	pEMC->removeEditMethod ( pEM ) ;
	DELETEP( pEM ) ;

	pEM = ev_EditMethod_lookup ( "s_abicollab_accounts" ) ;
	pEMC->removeEditMethod ( pEM ) ;
	DELETEP( pEM ) ;
	
	// now remove crap from the menus
	int frameCount = pApp->getFrameCount();
	XAP_Menu_Factory * pFact = pApp->getMenuFactory();

	pFact->removeMenuItem("Main", NULL, szCollaboration);
	pFact->removeMenuItem("Main", NULL, szCollaborationOffer);
	pFact->removeMenuItem("Main", NULL, szCollaborationJoin);
	pFact->removeMenuItem("Main", NULL, szCollaborationAccounts);	
	pFact->removeMenuItem("Main", NULL, szEndCollaboration);
	
	for (int i = 0; i < frameCount; ++i)
	{
		// Get the current frame that we're iterating through.
		XAP_Frame* pFrame = pApp->getFrame(i);
		pFrame->rebuildMenus();
	}
}

bool s_abicollab_offer(AV_View* v, EV_EditMethodCallData *d)
{
	UT_DEBUGMSG(("s_abicollab_offer\n"));
	AbiCollabSessionManager* pManager = AbiCollabSessionManager::getManager();
	XAP_Frame *pFrame = XAP_App::getApp()->getLastFocussedFrame();
	UT_return_val_if_fail(pFrame, false);
	PD_Document* pDoc = static_cast<PD_Document *>(pFrame->getCurrentDoc());
	UT_return_val_if_fail(pDoc, false);

 	if (pManager->isLocallyControlled(pDoc))
	{
		AbiCollab* pSession = pManager->getSessionFromDocumentId(pDoc->getDocUUIDString());
		if (pSession)
		{	
			pManager->closeSession(pSession, true);
		}
	}
	else
	{
		pManager->startSession(pDoc);
	}
	return true;
}

bool s_abicollab_join(AV_View* v, EV_EditMethodCallData *d)
{
	// Get the current view that the user is in.
	XAP_Frame *pFrame = XAP_App::getApp()->getLastFocussedFrame();
	// Get an Accounts dialog instance
	XAP_DialogFactory* pFactory = static_cast<XAP_DialogFactory *>(XAP_App::getApp()->getDialogFactory());
	UT_return_val_if_fail(pFactory, false);
	AP_Dialog_CollaborationJoin* pDialog = static_cast<AP_Dialog_CollaborationJoin*>(
				pFactory->requestDialog(AbiCollabSessionManager::getManager()->getDialogJoinId())
			);
	// Run the dialog
	pDialog->runModal(pFrame);
	pFactory->releaseDialog(pDialog);
	return true;
}


bool s_abicollab_accounts(AV_View* v, EV_EditMethodCallData *d)
{
	// Get the current view that the user is in.
	XAP_Frame *pFrame = XAP_App::getApp()->getLastFocussedFrame();
	// Get an Accounts dialog instance
	XAP_DialogFactory* pFactory = static_cast<XAP_DialogFactory *>(XAP_App::getApp()->getDialogFactory());
	UT_return_val_if_fail(pFactory, false);
	AP_Dialog_CollaborationAccounts* pDialog = static_cast<AP_Dialog_CollaborationAccounts*>(
				pFactory->requestDialog(AbiCollabSessionManager::getManager()->getDialogAccountsId())
			);
	// Run the dialog
	pDialog->runModal(pFrame);
	pFactory->releaseDialog(pDialog);
	return true;
}
