/* -*- mode: C++; tab-width: 4; c-basic-offset: 4; -*- */

/* AbiCollab- Code to enable the modification of remote documents.
 * Copyright (C) 2005 by Martin Sevior
 * Copyright (C) 2006 by Marc Maurer <uwog@uwog.net>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA
 * 02111-1307, USA.
 */

#include <string>
#include <vector>

#include "pd_Document.h"
#include "px_CR_SpanChange.h"
#include "px_CR_FmtMarkChange.h"  
#include "px_CR_SpanChange.h"
#include "px_CR_FmtMark.h"        
#include "px_CR_Span.h"
#include "px_CR_Glob.h"           
#include "px_CR_StruxChange.h"
#include "px_CR_ObjectChange.h"   
#include "px_CR_Strux.h"
#include "px_CR_Object.h"
#include "xap_App.h"
#include "pd_Style.h"
#include "xap_Frame.h"
#include "fv_View.h"
#include "ut_sleep.h"
#include "pt_Types.h"

#include "AbiCollab_Import.h"
#include "AbiCollab.h"
#include "AbiCollab_Export.h"
#include <backends/xp/Buddy.h>

ABI_Collab_Import::ABI_Collab_Import(AbiCollab* pAbiCollab, PD_Document* doc):
	m_pDoc(doc),
	m_pAbiCollab(pAbiCollab),
	m_iAlreadyRevertedRev(-1)
{
}

ABI_Collab_Import::~ABI_Collab_Import()
{
}

bool ABI_Collab_Import::_isOverlapping(UT_sint32 pos1, UT_sint32 length1, UT_sint32 pos2, UT_sint32 length2)
{
	if (pos1 == pos2)
		return true;
	else if (pos1 < pos2)
		return (pos1 + length1 - 1 >= pos2);
	else
		return (pos2 + length2 - 1 >= pos1);
}

void ABI_Collab_Import::_calculateCollisionSeqence(UT_sint32 iIncomingRemoteRev, const UT_UTF8String& sIncomingDocUUID, UT_sint32& iStart, UT_sint32& iEnd)
{
	UT_DEBUGMSG(("ABI_Collab_Import::_calculateCollisionSeqence() - iIncomingRemoteRev: %d\n", iIncomingRemoteRev));

	// initialization
	iStart = -1;
	iEnd = -1;

	ABI_Collab_Export* pExport = m_pAbiCollab->getExport();
	UT_return_if_fail(pExport);

	const UT_GenericVector<ChangeAdjust *>* pExpAdjusts = pExport->getAdjusts();
	UT_return_if_fail(pExpAdjusts);

	// worst case: the whole outgoing changerecord stack is the collision sequence
	iStart = 0;
	iEnd = pExpAdjusts->getItemCount();

	// scan back to find the changerecord in our export list the remote has seen,
	// maybe we can narrow the collision sequence down
	UT_sint32 i = 0;
	for (i = pExpAdjusts->getItemCount()-1; i >= 0; i--)
	{
	    ChangeAdjust * pChange = pExpAdjusts->getNthItem(i);
		if (pChange)
		{
		    UT_DEBUGMSG(("Looking at exported changerecord - rev: %d, pos: %d, length: %d adjust %d, queue pos: %d\n", 
						pChange->m_iCRNumber, pChange->m_iDocPos, pChange->m_iLength, pChange->m_iAdjust, i));

		    if (iIncomingRemoteRev >= pChange->m_iCRNumber)
		    {
				UT_DEBUGMSG(("Found the changerecord the remote side has already seen (iIncomingRemoteRev = %d)\n", iIncomingRemoteRev));

				// a new changerecord can't collide with a changerecord the 
				// remote side had already seen, hence the +1
				iStart = i+1;
				break;
		    }
		}
		else
			UT_ASSERT_HARMLESS(UT_SHOULD_NOT_HAPPEN);
	}

	// now move upward, so we kill off the bottom where changes are from the same document as the incoming cr (you can not collide with your own cr's!)
	for (; iStart < pExpAdjusts->getItemCount(); iStart++)
	{
		ChangeAdjust * pChange = pExpAdjusts->getNthItem(iStart);
		if (strcmp(pChange->m_pDocUUID, sIncomingDocUUID.utf8_str()) != 0)
		{
			// not the same document anymore, we can stop
			break;
		}
		else
			UT_DEBUGMSG(("Killing off matching change: %d\n", iStart));
	}
}

UT_sint32 ABI_Collab_Import::_getIncomingAdjustmentForState(const UT_GenericVector<ChangeAdjust *>* pExpAdjusts, UT_sint32 iStart, UT_sint32 iEnd, UT_sint32 iIncomingPos, UT_sint32 iIncomingLength, const UT_UTF8String& sIncomingUUID)
{
	UT_DEBUGMSG(("ABI_Collab_Import::_getIncomingAdjustmentForState()\n"));
	UT_return_val_if_fail(pExpAdjusts, 0);

	UT_sint32 iAdjust = 0;
	for (UT_sint32 j = iEnd-1; j>=iStart; j--)
	{
		ChangeAdjust* pPrev = pExpAdjusts->getNthItem(j);
		if (strcmp(sIncomingUUID.utf8_str(), pPrev->m_pDocUUID) == 0)
		{
			if (pPrev->m_iAdjust > 0) // TODO: is this check correct?
			{
				UT_DEBUGMSG(("Looking at possible adjustment with queue pos: %d, -adjust: %d\n", j, -pPrev->m_iAdjust));
				if (pPrev->m_iAdjust > 0 && _isOverlapping(pPrev->m_iOrigDocPos, pPrev->m_iLength, iIncomingPos+iAdjust, iIncomingLength)) // if the position was in the middle of an insert done previously, then we only need to take the insertion adjust partially into account
				{
					UT_DEBUGMSG(("ADJUST OVERLAP DETECTED with queue pos: %d, pPrev->m_iOrigDocPos: %d, pPrev->m_iLength: %d, iIncomingPos: %d, iAdjust: %d\n", j, pPrev->m_iOrigDocPos, pPrev->m_iLength, iIncomingPos, iAdjust));
					iAdjust -= (iIncomingPos+iAdjust - pPrev->m_iOrigDocPos);
				}
				else if (pPrev->m_iOrigDocPos <= iIncomingPos+iAdjust)
				{
					UT_DEBUGMSG(("ADJUSTMENT influenced normally by queue pos: %d\n", j));
					iAdjust -= pPrev->m_iAdjust;
				}
				else
					UT_DEBUGMSG(("no ADJUSTMENT influence by queue pos: %d, pPrev->m_iProgDocPos: %d, pPrev->m_iOrigDocPos\n", j, pPrev->m_iOrigDocPos, pPrev->m_iOrigDocPos));
			}
		}
	}
	return iAdjust;
}

/*!
 * Scan back through the CR's we've emitted since this remote CR was sent
 * and see if any overlap this one.
 * return true if there is a collision.
 */
bool ABI_Collab_Import::_checkForCollision(UT_sint32 iIncomingPos, UT_sint32 iIncomingLength, UT_sint32 iIncomingAdjust, const UT_UTF8String& UUID, UT_sint32 iIncomingRemoteRev, UT_sint32& iRev, UT_sint32& iImportAdjustment)
{
	UT_DEBUGMSG(("ABI_Collab_Import::_checkForCollision() - iIncomingPos: %d, iIncomingLength: %d, UUID: %s, iIncomingRemoteRev: %d\n", iIncomingPos, iIncomingLength, UUID.utf8_str(), iIncomingRemoteRev));

	ABI_Collab_Export* pExport = m_pAbiCollab->getExport();
	UT_return_val_if_fail(pExport, false);

	const UT_GenericVector<ChangeAdjust *>* pExpAdjusts = pExport->getAdjusts();
	UT_return_val_if_fail(pExpAdjusts, false);

	iImportAdjustment = 0;

	// get the collision sequence (if any)
	UT_sint32 iStart = 0;
	UT_sint32 iEnd = 0;
	_calculateCollisionSeqence(iIncomingRemoteRev, UUID, iStart, iEnd);
	UT_return_val_if_fail(iStart >= 0 && iEnd >= 0, false);
	if (iStart == iEnd)
	{
		UT_DEBUGMSG(("Empty collision sequence, no possible collision\n"));
		return false;
	}

	UT_sint32 iIncomingStateAdjust = _getIncomingAdjustmentForState(pExpAdjusts, iStart, iEnd, iIncomingPos, iIncomingLength, UUID);
	UT_DEBUGMSG(("IINCOMMINGSTATEADJUST: %d\n", iIncomingStateAdjust));
	UT_sint32 iIncomingStateAdjustUp = 0;

	// Now scan forward and look for an overlap of the new changerecord with the collision sequence
	UT_DEBUGMSG(("Checking collision sequence [%d..%d) for overlapping changerecords\n", iStart, iEnd));
	bool bDenied = false;
	for (UT_sint32 i = iStart; i < iEnd; i++)
	{
	    ChangeAdjust* pChange = pExpAdjusts->getNthItem(i);
		if (pChange)
		{
			UT_DEBUGMSG(("Looking at pChange->m_pDocUUID: %s\n", pChange->m_pDocUUID));

			UT_sint32 iLocalStateAdjust = 0;
			UT_DEBUGMSG(("ILOCALSTATEADJUST for pos %d: %d\n", i, iLocalStateAdjust));

			if (strcmp(pChange->m_pDocUUID, UUID.utf8_str()) != 0)
			{
				if (_isOverlapping(iIncomingPos+iIncomingStateAdjust+iIncomingStateAdjustUp, iIncomingLength, pChange->m_iDocPos+iLocalStateAdjust, pChange->m_iLength))
				{
					bDenied = true;

					//
					// some overlaps are just fine; let's special case some here
					//

					if (iIncomingAdjust > 0 && pChange->m_iAdjust > 0)
					{
						// overlapping inserts are just fine, as long as the start positions differ
						if (iIncomingPos+iIncomingStateAdjust+iIncomingStateAdjustUp != pChange->m_iDocPos)
						{
							UT_DEBUGMSG(("ALLOWING OVERLAPPING CHANGERECORD AFTER ALL\n"));
							bDenied = false;
						}
					}

					// TODO: add more overlaps, for example ones that change only (non-conflicting) formatting marks
				}

				if (bDenied)
				{
					UT_DEBUGMSG(("Overlap detected for iIncomingPos: %d, iLocalStateAdjust: %d, iIncomingLength: %d, pChange->m_iDocPos: %d, pChange->m_iLength: %d\n", iIncomingPos, iLocalStateAdjust, iIncomingLength, pChange->m_iDocPos, pChange->m_iLength));
					iRev = pChange->m_iCRNumber;
					break;
				}
				else
					UT_DEBUGMSG(("No overlap detected for iIncomingPos: %d, iLocalStateAdjust: %d, iIncomingLength: %d, pChange->m_iDocPos: %d, pChange->m_iLength: %d\n", iIncomingPos, iLocalStateAdjust, iIncomingLength, pChange->m_iDocPos, pChange->m_iLength));
			}
			else
				UT_DEBUGMSG(("Skipping overlap detection: changerecords came from the same document; iIncomingPos: %d, iIncomingAdjust: %d, iIncomingLength: %d, pChange->m_iDocPos: %d, pChange->m_iLength: %d\n", iIncomingPos, iIncomingAdjust, iIncomingLength, pChange->m_iDocPos, pChange->m_iLength));

			if (pChange->m_iDocPos+iLocalStateAdjust < iIncomingPos+iIncomingStateAdjust+iIncomingStateAdjustUp)
			{
				UT_DEBUGMSG(("Normal Upward influence detected\n"));
				iIncomingStateAdjustUp += pChange->m_iAdjust;
			}
			else if (pChange->m_iDocPos+iLocalStateAdjust == iIncomingPos+iIncomingStateAdjust+iIncomingStateAdjustUp)
			{
				if (strcmp(pChange->m_pDocUUID, UUID.utf8_str()) == 0)
				{
					if (pChange->m_iOrigDocPos <= iIncomingPos+iIncomingStateAdjust+iIncomingStateAdjustUp)
					{
						UT_DEBUGMSG(("Normal upward local influence\n"));
						iIncomingStateAdjustUp += pChange->m_iAdjust; // FIXME: niet voor de hele adjustment als length > 1
					}
				}
				else
				{
					if (!bDenied)
					{
						UT_ASSERT_HARMLESS(UT_SHOULD_NOT_HAPPEN); // this should have been a collition
					}
					else
					{
						UT_DEBUGMSG(("Skipping upward adjustment; collision detected\n"));
					}
				}
			}
			UT_DEBUGMSG(("iIncomingStateAdjustUp is now: %d\n", iIncomingStateAdjustUp));
		}
		else
			UT_return_val_if_fail(false, false);
	}

	iImportAdjustment = iIncomingStateAdjust+iIncomingStateAdjustUp;
	UT_DEBUGMSG(("Full import adjustment: %d\n", iImportAdjustment));

	return bDenied;
}

// returns true if the import can continue, false otherwise
bool ABI_Collab_Import::_handleCollision(UT_sint32 iIncomingRev, UT_sint32 iLocalRev, const Buddy& collaborator)
{
	UT_DEBUGMSG(("_handleCollision() - incoming rev %d collides against local rev %d!!!\n", iIncomingRev, iLocalRev));

	if (m_pAbiCollab->isLocallyControlled())
	{
		UT_DEBUGMSG(("We're controlling this session, refusing this changerecord from %s!\n", collaborator.getName().utf8_str()));
		// add this collaborator to our revert ack list, so we can ignore his packets
		// until we get an acknoledgement that he has reverted his local, colliding changes
		m_revertSet.push_back(collaborator.getName());
		// send the revert command to the collaborator
		RevertSessionPacket rsp(m_pAbiCollab->getSessionId(), iIncomingRev);
		const UT_UTF8String* psPacket = rsp.serialize();
		m_pAbiCollab->push(*psPacket, collaborator);
		return false;
	}
	else
	{
		UT_DEBUGMSG(("We're NOT controlling this session, reverting local changes and accepting changerecord!\n"));

		ABI_Collab_Export* pExport = m_pAbiCollab->getExport();
		UT_return_val_if_fail(pExport, false);

		UT_GenericVector<ChangeAdjust *>* pExpAdjusts = pExport->getAdjusts();
		UT_return_val_if_fail(pExpAdjusts, false);
		
		m_pAbiCollab->setIsReverting(true); // mask all changes in the exporter

		// undo our cool local changes, and nuke our exported packet list as well up to (and including) iLocalRev
		for (UT_sint32 i = pExpAdjusts->getItemCount() - 1; i >= 0; i--)
		{
			ChangeAdjust* pChange = pExpAdjusts->getNthItem(i);
			if (pChange)
			{
				if (pChange->m_iCRNumber >= iLocalRev)
				{
					if (strcmp(m_pDoc->getOrigDocUUIDString(), pChange->m_pDocUUID) == 0)
					{
						UT_DEBUGMSG(("UNDO-ING AND NUKING LOCAL CHANGE: EXPORT POSITION %d, pChange->m_iCRNumber: %d!\n", i, pChange->m_iCRNumber));

						m_pDoc->undoCmd(1);
						UT_sint32 iTmpRev = pChange->m_iCRNumber;
						delete pChange;
						pExpAdjusts->deleteNthItem(i);

						// FIXME FIXME FIXME:
						UT_DEBUGMSG(("***********************************************************!\n")); // really? i can't seem to trigger it anymore
						UT_DEBUGMSG(("***********************************************************!\n")); // really? i can't seem to trigger it anymore
						UT_DEBUGMSG(("***********************************************************!\n")); // really? i can't seem to trigger it anymore
						UT_DEBUGMSG(("***********************************************************!\n")); // really? i can't seem to trigger it anymore
						UT_DEBUGMSG(("TODO: implement me: fixup all changes above the nuke point!\n")); // really? i can't seem to trigger it anymore
						UT_DEBUGMSG(("***********************************************************!\n")); // really? i can't seem to trigger it anymore
						UT_DEBUGMSG(("***********************************************************!\n")); // really? i can't seem to trigger it anymore
						UT_DEBUGMSG(("***********************************************************!\n")); // really? i can't seem to trigger it anymore
						UT_DEBUGMSG(("***********************************************************!\n")); // really? i can't seem to trigger it anymore
					}
					else
						UT_DEBUGMSG(("Skipping undo of remote change\n"));
				}
				else
					break;
			}
			else
				UT_ASSERT_HARMLESS(UT_SHOULD_NOT_HAPPEN);
		}

		m_pAbiCollab->setIsReverting(false); // unmask all changes in the exporter

		m_iAlreadyRevertedRev = iLocalRev;

		UT_DEBUGMSG(("Pre-Acknowledging revert of revision %d\n", iLocalRev));
		// send the revert acknowledgement command to the session owner
		RevertAckSessionPacket rasp(m_pAbiCollab->getSessionId(), iLocalRev);
		const UT_UTF8String* psPacket = rasp.serialize();
		m_pAbiCollab->push(*psPacket, collaborator);

		return true;
	}
}

bool ABI_Collab_Import::_shouldIgnore(const Buddy& collaborator)
{
	if (m_pAbiCollab->isLocallyControlled())
	{
		UT_DEBUGMSG(("This session is locally controlled, check if we are waiting for a revert ack from buddy: %s\n", collaborator.getName().utf8_str()));
		// see if we are waiting for a revert ack packet from this collaborator;
		// if we do, then just drop all packets on the floor until we see it
		for (vector<UT_UTF8String>::iterator it = m_revertSet.begin(); it != m_revertSet.end(); it++)
		{
			if (*it == collaborator.getName())
			{
				UT_DEBUGMSG(("Found collaborator %s on our revert ack list; changerecords should be ignored!\n", collaborator.getName().utf8_str()));
				return true;
			}
		}
	}
	UT_DEBUGMSG(("%s is not on our revert ack list, don't ignore this packet...\n", collaborator.getName().utf8_str()));
	return false;
}

bool ABI_Collab_Import::import(const SessionPacket& packet, const Buddy& collaborator)
{
	UT_DEBUGMSG(("ABI_Collab_Import::import()\n"));

	// check for collisions to see if we can import this packet at all;
	// NOTE: the position adjustment is calculated in the process, so we don't have to 
	// do that again on import (as calculating it for globs can be quite costly, and
	// we need to do it anyway to properly detect collisions
	UT_sint32 iImportAdjustment = 0;
	switch (packet.getSessionType())
	{
		case PACKET_GLOB:
			{
				if (_shouldIgnore(collaborator))
					return false;

				const GlobSessionPacket* gp = static_cast<const GlobSessionPacket*>(&packet);
				UT_return_val_if_fail(gp->getPackets().getItemCount() > 0, false);

				// determine the position and length of the whole glob combined,
				// so we can determine with 1 chech of this glob causes a collision
				// (instead of determining it for each individual changerecord in this glob)
				UT_sint32 iGlobPos = 0;
				UT_sint32 iGlobLength = 0;
				UT_sint32 iGlobAdjust = 0;
				UT_sint32 iGlobRev = static_cast<const ChangeRecordSessionPacket*>(gp->getPackets().getNthItem(0))->getRev();
				UT_sint32 iGlobRemoteRev = static_cast<const ChangeRecordSessionPacket*>(gp->getPackets().getNthItem(0))->getRemoteRev();
				const UT_UTF8String& iGlobDocUUID = static_cast<const ChangeRecordSessionPacket*>(gp->getPackets().getNthItem(0))->getDocUUID();
				for (UT_sint32 j = 0; j < gp->getPackets().getItemCount(); j++)
				{
					SessionPacket* pGlobPacket = gp->getPackets().getNthItem(j);
					if (pGlobPacket->getSessionType() == PACKET_CHANGERECORD)
					{
						const ChangeRecordSessionPacket* crp = static_cast<const ChangeRecordSessionPacket*>(pGlobPacket);
						// HACK: try to determine the _real_ start of the glob
						// FIXME: this will include PXT_ChangePoint crap, which can screw stuff up I guess
						if (iGlobPos == 0)
							iGlobPos = crp->getPos();
						if (crp->getPos() > 0 && iGlobPos > crp->getPos()) 
							iGlobPos = crp->getPos();
						iGlobLength += crp->getLength();
						iGlobAdjust += crp->getAdjust();
					}
				}
				UT_DEBUGMSG(("Determining collision for glob as a whole: iGlobPos: %d, iGlobLength: %d, iGlobRemoteRev: %d, iGlobRev: %d\n", iGlobPos, iGlobLength, iGlobRemoteRev, iGlobRev));
				UT_ASSERT_HARMLESS(iGlobPos > 0 && iGlobLength > 0); // GLOBs with length 0 can happen I guess, but for now, it doesn't make sense

				UT_sint32 iLocalRev = 0;
				bool bCollide = _checkForCollision(iGlobPos, iGlobLength, iGlobAdjust, iGlobDocUUID, iGlobRemoteRev, iLocalRev, iImportAdjustment);
				// make sure we revert all the way upto the first revision in this glob if a collision is detected
				bool continueImport = (bCollide	? _handleCollision(iGlobRev, iLocalRev, collaborator) : true);
				if (!continueImport)
					return false;	

				m_pAbiCollab->setOrigDocPos(iGlobPos); // HACK HACK HACK
				m_pAbiCollab->setOrigRemoteSeen(iGlobRemoteRev); // HACK HACK HACK
			}
			break;
		case PACKET_CHANGERECORD:
			{
				if (_shouldIgnore(collaborator))
					return false;

				// check for a collision and handle it if it occurred
				const ChangeRecordSessionPacket* crp = static_cast<const ChangeRecordSessionPacket*>(&packet);
				UT_sint32 iLocalRev = 0;
				bool bCollide = _checkForCollision(crp->getPos(), crp->getLength(), crp->getAdjust(), crp->getDocUUID(), crp->getRemoteRev(), iLocalRev, iImportAdjustment);
				bool continueImport = (bCollide ? _handleCollision(crp->getRev(), iLocalRev, collaborator) : true);
				if (!continueImport)
					return false;

				m_pAbiCollab->setOrigDocPos(crp->getPos()); // HACK HACK HACK
				m_pAbiCollab->setOrigRemoteSeen(crp->getRemoteRev()); // HACK HACK HACK
			}
			break;

		case PACKET_SIGNAL:
		case PACKET_REVERT:
		case PACKET_REVERT_ACK:
			// these packets can never cause collisions, let them pass
			break;
		default:
			UT_ASSERT_HARMLESS(UT_NOT_IMPLEMENTED);
			break;
	}

	UT_DEBUGMSG(("Collision test succeeded, continuing to import this packet\n"));

	UT_GenericVector<AV_View *> vecViews;
	m_pDoc->getAllViews(&vecViews);

	for (UT_sint32 i=0; i < vecViews.getItemCount(); i++)
	{
		vecViews.getNthItem(i)->setActivityMask(false);
	}

	// import the changes in the document
	UT_UTF8String sRealDocname = m_pDoc->getOrigDocUUIDString();
	bool bRes = _import(packet, iImportAdjustment, collaborator);
	m_pDoc->setMyUUID(sRealDocname.utf8_str());
	UT_DEBUGMSG(("Resetting orig docpos && orig remote seen\n"));
	m_pAbiCollab->setOrigDocPos(-1); // HACK HACK HACK
	m_pAbiCollab->setOrigRemoteSeen(-1); // HACK HACK HACK

	bRes = true; // force nice updates for now; stuff like tables still report false on import, due to formatting marks without props/attrs
	if (bRes)
	{
		bool bDone = false;
		for (UT_sint32 i = 0; i<vecViews.getItemCount(); i++)
		{
			FV_View * pView = static_cast<FV_View *>( vecViews.getNthItem(i));
			if(pView && !bDone && pView->shouldScreenUpdateOnGeneralUpdate())
			{
				m_pDoc->signalListeners(PD_SIGNAL_UPDATE_LAYOUT);
				bDone = true;
			}
			if(pView)
			{
				pView->fixInsertionPointCoords();
				pView->setActivityMask(true);
			}
		}
	}
	else
	{
		for (UT_sint32 i = 0; i<vecViews.getItemCount(); i++)
		{
			vecViews.getNthItem(i)->setActivityMask(true);
		}
	}

	return bRes;
}

/*!
 * Take a packet contained with a UT_UTF8string, interpret it's
 * contents and apply the implied operations on the document.
 */
bool ABI_Collab_Import::_import(const SessionPacket& packet, UT_sint32 iImportAdjustment, const Buddy& collaborator, bool inGlob)
{
	UT_DEBUGMSG(("ABI_Collab_Import::_import() - packet type: %d, iImportAdjustment: %d\n", packet.getSessionType(), iImportAdjustment));

	switch (packet.getSessionType())
	{
		case PACKET_GLOB:
			{
				const GlobSessionPacket* gp = static_cast<const GlobSessionPacket*>(&packet);
				UT_return_val_if_fail(gp->getPackets().getItemCount() > 0, false);

				// store the last seen revision from this collaborator (it is immediately used by the export)
				UT_sint32 iGlobRev = static_cast<const ChangeRecordSessionPacket*>(gp->getPackets().getNthItem(0))->getRev();
				m_remoteRevs[collaborator.getName().utf8_str()] = iGlobRev;

				// lock out all updates while processing this glob
				m_pDoc->notifyPieceTableChangeStart();
				m_pDoc->disableListUpdates();
				m_pDoc->setDontImmediatelyLayout(true);
				m_pDoc->beginUserAtomicGlob();

				for (UT_sint32 j = 0; j < gp->getPackets().getItemCount(); j++)
				{
					SessionPacket* pGlobPacket = gp->getPackets().getNthItem(j);
					if (pGlobPacket)
					{
						bool res = _import(*pGlobPacket, iImportAdjustment, collaborator, true); // yay for recursion :)
						if (!res)
							UT_ASSERT_HARMLESS(UT_SHOULD_NOT_HAPPEN);
					}
				}

				// allow updates again
				m_pDoc->enableListUpdates();
				m_pDoc->updateDirtyLists();
				m_pDoc->notifyPieceTableChangeEnd();
				m_pDoc->setDontImmediatelyLayout(false);
				m_pDoc->endUserAtomicGlob();

				return true;
			}
		
		case PACKET_SIGNAL:
			{
				const SignalSessionPacket* sp = static_cast<const SignalSessionPacket*>(&packet);
				// TODO: FIX ME FIX ME FIXME
				if(0 /*m_pAbiCollab->isServer() */ )
				{
					if( sp->getSignalType() == PD_SIGNAL_DOCSAVED)
					{
						m_pDoc->save();
						return true;
					}
				}
				if (sp->getSignalType() == PD_SIGNAL_DOCCLOSED)
				{
					m_pAbiCollab->setClose();
					UT_DEBUGMSG(("Doing close from remote signal \n"));
					return true;
				}
				m_pDoc->signalListeners(sp->getSignalType());
				return true;
			}

		case PACKET_REVERT:
			{
				const RevertSessionPacket* rrp = static_cast<const RevertSessionPacket*>(&packet);
				UT_DEBUGMSG(("Revert packet seen on import for rev: %d\n", rrp->getRev()));

				if (m_iAlreadyRevertedRev == -1 || m_iAlreadyRevertedRev != rrp->getRev())
				{
					UT_DEBUGMSG(("Incoming revert for revision %d, which we didn't detect locally (m_iAlreadyRevertedRev: %d)!\n", rrp->getRev(), m_iAlreadyRevertedRev));
					UT_DEBUGMSG(("DOCUMENT OUT OF SYNC DETECTED!!!!\n"));
					UT_ASSERT_HARMLESS(UT_NOT_IMPLEMENTED);
					return false;
				}	
					
				m_iAlreadyRevertedRev = -1;
				return true;
			}
		case PACKET_REVERT_ACK:
			{
				const RevertAckSessionPacket* rarp = static_cast<const RevertAckSessionPacket*>(&packet);
				UT_DEBUGMSG(("RevertAck packet seen on import for rev: %d\n", rarp->getRev()));

				// TODO: check the rev to make sure it matches the one we sent out

				// remove this collaborator from our revert ack list; he can play again...
				for (vector<UT_UTF8String>::iterator it = m_revertSet.begin(); it != m_revertSet.end(); it++)
				{
					if (*it == collaborator.getName())
					{
						UT_DEBUGMSG(("Found collaborator %s on our revert ack list! Removing him from the list...\n", (*it).utf8_str()));
						m_revertSet.erase(it);
						return true;
					}
				}

				UT_ASSERT_HARMLESS(UT_SHOULD_NOT_HAPPEN);
				return false;
			}

		case PACKET_CHANGERECORD:
			{
				const ChangeRecordSessionPacket* crp = static_cast<const ChangeRecordSessionPacket*>(&packet);
				UT_DEBUGMSG(("It's safe to import this packet\n"));

				UT_sint32 iRev = crp->getRev();
				UT_DEBUGMSG(("For CR number %d requested point %d adjustment %d \n", iRev, crp->getPos(), iImportAdjustment));

				PT_DocPosition pos = static_cast<PT_DocPosition>(crp->getPos() + iImportAdjustment);
				UT_ASSERT(pos <= getEndOfDoc());

				if (!inGlob)
				{
					// store the last seen revision from this collaborator (it is immediately used by the export)
					// NOTE: if this changerecord is part of a glob, then we don't do this; we'll have
					// already set the revision of the glob itself as the last seen one
					m_remoteRevs[collaborator.getName().utf8_str()] = crp->getRev();
				}

				// todo: remove these temp vars
				UT_UTF8String sFrag = crp->getFrag();
				const gchar** szProps = const_cast<const gchar**>(crp->getProps());
				const gchar** szAtts = const_cast<const gchar**>(crp->getAtts());
				UT_UTF8String sValue = crp->getValue();
				PT_DocPosition iPos2 = 0;

				m_pDoc->setMyUUID(crp->getDocUUID().utf8_str());

				// process the packet
				switch(crp->getPXType())
				{
					case PX_ChangeRecord::PXT_GlobMarker:
						UT_DEBUGMSG(("Found GLOB marker (ignoring)\n"));
						return true;
					case PX_ChangeRecord::PXT_InsertSpan:
						{
							UT_DEBUGMSG(("Doing InsertSpan |%s| \n", sValue.utf8_str()));
							UT_UCS4String UCSChars = sValue.ucs4_str();
							m_pDoc->insertSpan(pos,UCSChars.ucs4_str(),UCSChars.length());
						}
						break;
					case PX_ChangeRecord::PXT_DeleteSpan:
						{
							iPos2 = pos + crp->getLength();
							PP_AttrProp *p_AttrProp_Before = NULL;
							UT_uint32 icnt = 0;
							m_pDoc->deleteSpan(pos,iPos2,p_AttrProp_Before,icnt,true);
						}
						break;
					case PX_ChangeRecord::PXT_ChangeSpan:
						{
							if (sFrag == "span")
							{
								iPos2 = pos + crp->getLength();
								if((szProps == NULL) && (szAtts == NULL))
								{
									//
									// This happens if we remove all formats
									// we have to handle this seperately
									//
									// Get style of containing block
									//
									PL_StruxDocHandle sdh = NULL;
									m_pDoc->getStruxOfTypeFromPosition(pos,PTX_Block,&sdh);
									if(!sdh)
									{
										UT_ASSERT(UT_SHOULD_NOT_HAPPEN);
										return false;
									}
									PD_Style * pStyle = m_pDoc->getStyleFromSDH(sdh);
									if(!pStyle)
									{
										UT_ASSERT(UT_SHOULD_NOT_HAPPEN);
										return false;
									}
									const char * szName =  pStyle->getName();
									const char * atts[3] = {PT_STYLE_ATTRIBUTE_NAME,szName,NULL};
									m_pDoc->changeSpanFmt(PTC_SetExactly, pos, iPos2, atts, szProps);
									break;
								}
								printf(" 0 |%s| 1 |%s| 2 |%s| \n", szProps[0], szProps[0], szProps[2]);
								m_pDoc->changeSpanFmt(PTC_SetExactly, pos, iPos2, szAtts, szProps);
							}
							else
							{
								return false;
							}
						}
						break;
					case PX_ChangeRecord::PXT_InsertStrux:
						{
							UT_DEBUGMSG(("AbiCollab -- Doing insertStrux pos %d frag |%s|\n",pos,sFrag.utf8_str()));
							PTStruxType pts;
							if (sFrag == "p")
							{
								pts = PTX_Block;
							}
							else if (sFrag == "section")
							{
								pts = PTX_Section;
							}
							else if (sFrag == "hdrftr")
							{
								pts = PTX_SectionHdrFtr;
							}
							else if (sFrag == "footnote")
							{
								pts = PTX_SectionFootnote;
							}
							else if (sFrag == "endfootnote")
							{
								pts = PTX_EndFootnote;
							}
							else if (sFrag == "toc")
							{
								pts = PTX_SectionTOC;
							}
							else if (sFrag == "endtoc")
							{
								pts = PTX_EndTOC;
							}
							else if (sFrag == "endnote")
							{
								pts = PTX_SectionEndnote;
							}
							else if(sFrag == "endendnote")
							{
								pts = PTX_EndEndnote;
							}
							else if (sFrag == "table")
							{
								pts = PTX_SectionTable;
							}
							else if (sFrag == "endtable")
							{
								pts = PTX_EndTable;
							}
							else if (sFrag == "cell")
							{
								pts = PTX_SectionCell;
							}
							else if (sFrag == "endcell")
							{
								pts = PTX_EndCell;
							}
							else if (sFrag == "frame")
							{
								pts = PTX_SectionFrame;
							}
							else if (sFrag == "endframe")
							{
								pts = PTX_EndFrame;
							}
							else
							{
								UT_DEBUGMSG(("Unimplemented sFrag: %s\n", sFrag.utf8_str()));
								UT_ASSERT_HARMLESS(UT_NOT_IMPLEMENTED);
							}
							
							if((szProps != NULL) || (szAtts != NULL))
							{
							        m_pDoc->insertStrux(pos, pts, szAtts,szProps);
							}
							else
							{
								m_pDoc->insertStrux(pos, pts);
							}
						}
						break;
					case PX_ChangeRecord::PXT_DeleteStrux:
						{
							PTStruxType pts;
							if(sFrag== "p")
							{
								pts = PTX_Block;
							}
							else if (sFrag == "section")
							{
								pts = PTX_Section;
							}
							else if (sFrag == "footnote")
							{
								pts = PTX_SectionFootnote;
							}
							else if (sFrag == "endfootnote")
							{
								pts = PTX_EndFootnote;
							}
							else if (sFrag == "toc")
							{
								pts = PTX_SectionTOC;
							}
							else if (sFrag == "endtoc")
							{
								pts = PTX_EndTOC;
							}
							else if (sFrag == "endnote")
							{
								pts = PTX_SectionEndnote;
							}
							else if (sFrag == "endendnote")
							{
								pts = PTX_EndEndnote;
							}
							else if (sFrag == "table")
							{
								pts = PTX_SectionTable;
							}
							else if (sFrag == "endtable")
							{
								pts = PTX_EndTable;
							}
							else if (sFrag == "cell")
							{
								pts = PTX_SectionCell;
							}
							else if (sFrag == "endcell")
							{
								pts = PTX_EndCell;
							}
							else if (sFrag == "frame")
							{
								pts = PTX_SectionFrame;
							}
							else if (sFrag == "endframe")
							{
								pts = PTX_EndFrame;
							}
							else
							{
								UT_DEBUGMSG(("Unimplemented sFrag: %s\n", sFrag.utf8_str()));
								UT_ASSERT_HARMLESS(UT_NOT_IMPLEMENTED);
							}

							m_pDoc->deleteStrux(pos,pts,true);
						}
						break;
					case PX_ChangeRecord::PXT_ChangeStrux:
						{
							PTStruxType pts;
							if(sFrag == "p")
							{
								pts = PTX_Block;
							}
							else if(sFrag == "section")
							{
								pts = PTX_Section;
							}
							else if(sFrag == "footnote")
							{
								pts = PTX_SectionFootnote;
							}
							else if(sFrag == "endfootnote")
							{
								pts = PTX_EndFootnote;
							}
							else if( sFrag == "toc")
							{
								pts = PTX_SectionTOC;
							}
							else if( sFrag == "endtoc")
							{
								pts = PTX_EndTOC;
							}
							else if(sFrag == "endnote")
							{
								pts = PTX_SectionEndnote;
							}
							else if(sFrag == "endendnote")
							{
								pts = PTX_EndEndnote;
							}
							else if(sFrag == "table")
							{
								pts = PTX_SectionTable;
							}
							else if(sFrag == "endtable")
							{
								pts = PTX_EndTable;
							}
							else if(sFrag == "cell")
							{
								pts = PTX_SectionCell;
							}
							else if(sFrag == "endcell")
							{
								pts = PTX_EndCell;
							}
							else if(sFrag == "frame")
							{
								pts = PTX_SectionFrame;
							}
							else if(sFrag == "endframe")
							{
								pts = PTX_EndFrame;
							}
							else
							{
								UT_DEBUGMSG(("Unimplemented sFrag: %s\n", sFrag.utf8_str()));
								UT_ASSERT_HARMLESS(UT_NOT_IMPLEMENTED);
							}							

							UT_return_val_if_fail(szProps != NULL || szAtts != NULL, false);

							UT_DEBUGMSG(("Executing ChangeStrux pos= %d \n",pos));
							m_pDoc->changeStruxFmt(PTC_SetExactly, pos, pos, szAtts, szProps, pts);
							
							// TODO: this mask is waaaay to generic
							XAP_Frame *pFrame = XAP_App::getApp()->getLastFocussedFrame();
							FV_View* pView = static_cast<FV_View*>(pFrame->getCurrentView());
							pView->notifyListeners(AV_CHG_TYPING | AV_CHG_FMTCHAR | AV_CHG_FMTBLOCK | AV_CHG_PAGECOUNT | AV_CHG_FMTSTYLE );
						}
						break;
					case PX_ChangeRecord::PXT_InsertObject:
						{
							PTObjectType pto;
							if(sFrag == "image")
							{
								pto = PTO_Image;
								UT_sint32 i = 0;
								for(i=0;szProps[i] != NULL;i++)
								{
									UT_DEBUGMSG(("Image: szProps = |%s| \n",szProps[i]));
								}
							}
							else if(sFrag == "field")
							{
								pto = PTO_Field;
							}
							else if(sFrag == "bookmark")
							{
								pto = PTO_Bookmark;
							}
							else if(sFrag == "hyperlink")
							{
								pto = PTO_Hyperlink;
							}
							else if(sFrag == "math")
							{
								pto = PTO_Math;
							}
							else if(sFrag == "embed")
							{
								pto = PTO_Embed;
							}
							else
							{
								UT_DEBUGMSG(("Unimplemented sFrag: %s\n", sFrag.utf8_str()));
								UT_ASSERT_HARMLESS(UT_NOT_IMPLEMENTED);
							}
			
							if((szProps == NULL) && (szAtts == NULL))
							{
								UT_ASSERT(UT_SHOULD_NOT_HAPPEN);
								return false;
							}
							m_pDoc->insertObject(pos, pto, szAtts, szProps);
						}
						break;
					case PX_ChangeRecord::PXT_DeleteObject:
						{
							iPos2 = pos + 1;
							PP_AttrProp *p_AttrProp_Before = NULL;
							UT_uint32 icnt = 0;
							m_pDoc->deleteSpan(pos, iPos2, p_AttrProp_Before, icnt, true);
						}
						break;
					case PX_ChangeRecord::PXT_ChangeObject:
						{
							if ((szProps == NULL) && (szAtts == NULL))
							{
								UT_ASSERT(UT_SHOULD_NOT_HAPPEN);
								return false;
							}
							m_pDoc->changeSpanFmt(PTC_SetExactly, pos, pos + 1, szAtts, szProps);
						}
						break;
					case PX_ChangeRecord::PXT_InsertFmtMark:
						{
							if((szProps == NULL) && (szAtts == NULL))
							{
								// nothing to do here, please move along
								
								// NOTE: why does this happen anyway? 
								// This happens when for example when sending over tables:
								// <cr type="PXT_InsertFmtMark" docUUID="51779860-8d6a-11db-900e-a685fd336747" myUUID="66bf2a26-8d6a-11db-900e-a685fd336747" rev="5" remoterev="0" pos="4">
								//      <span/>
								// </cr>
								//UT_ASSERT_HARMLESS(UT_SHOULD_NOT_HAPPEN);
								return false;
							}
							m_pDoc->changeSpanFmt(PTC_SetExactly, pos, pos, szAtts, szProps);
						}
						break;
					case PX_ChangeRecord::PXT_DeleteFmtMark:
						{
							m_pDoc->deleteFmtMark(pos);
						}
						break; 
					case PX_ChangeRecord::PXT_ChangeFmtMark:
						{
							if ((szProps == NULL) && (szAtts == NULL))
							{
								UT_ASSERT(UT_SHOULD_NOT_HAPPEN);
								return false;
							}
							m_pDoc->changeSpanFmt(PTC_SetExactly, pos, pos, szAtts, szProps);
						}
						break;
					case PX_ChangeRecord::PXT_ChangePoint:
						UT_DEBUGMSG(("Change Point CR \n"));
						m_pDoc->createAndSendCR(pos, crp->getPXType(), true, 0);
						break; 
					case PX_ChangeRecord::PXT_ListUpdate:
						UT_DEBUGMSG(("ListUpdate CR \n"));
						m_pDoc->createAndSendCR(pos, crp->getPXType(), true,0);
						break; 
					case PX_ChangeRecord::PXT_StopList:
						UT_DEBUGMSG(("StopList CR \n"));
						m_pDoc->createAndSendCR(pos, crp->getPXType(), true,0);
						break; 
					case PX_ChangeRecord::PXT_UpdateField:
						UT_DEBUGMSG(("UpdateFiled CR \n"));
						m_pDoc->createAndSendCR(pos, crp->getPXType(), true,0);
						break;
					case PX_ChangeRecord::PXT_RemoveList:
						UT_DEBUGMSG(("RemoveList CR \n"));
						m_pDoc->createAndSendCR(pos, crp->getPXType(), true,0);
						break;
					case PX_ChangeRecord::PXT_UpdateLayout:
						UT_DEBUGMSG(("UpdateLayout CR \n"));
						m_pDoc->createAndSendCR(pos, crp->getPXType(), true,0);
						break;
					case PX_ChangeRecord::PXT_CreateDataItem:
						{
							UT_DEBUGMSG(("Doing CreateDataItem |%s| \n", sValue.utf8_str()));
							const char * szNameV = g_strdup(crp->getAttribute(PT_DATAITEM_ATTRIBUTE_NAME));
							const void * pToken = NULL;
							void * pHandle = NULL;
							UT_ByteBuf * pBuf= new UT_ByteBuf();
							pBuf->append(reinterpret_cast<const UT_Byte *>(sValue.utf8_str()),sValue.size());

							m_pDoc->createDataItem(szNameV,true,pBuf,pToken,&pHandle);
							delete pBuf;
						}
						break;
					default:
						UT_DEBUGMSG(("Unimplemented crp->getPXType(): %d\n", crp->getPXType()));
						UT_ASSERT_HARMLESS(UT_NOT_IMPLEMENTED);
						break;
				}
			}
			return true;			
		
		default:
			UT_DEBUGMSG(("ABI_Collab_Import::import called with wrong packet type: %d!\n", packet.getType()));
			return false;			
	}

	return false;
}

bool ABI_Collab_Import::getStruxType(UT_UTF8String & sInPacket, 
				     PTStruxType & struxType,
				     UT_UTF8String & sAttributes, 
				     UT_UTF8String & sOutpacket)
{
	return true;
}

bool ABI_Collab_Import::getObjectType(UT_UTF8String & sInPacket,
				      PTObjectType & objectType,
				      UT_UTF8String & sAttributes, 
				      UT_UTF8String & sOutpacket)
{
	return true;
}

PT_DocPosition ABI_Collab_Import::getEndOfDoc(void)
{
        PT_DocPosition posEnd;
        m_pDoc->getBounds(true,posEnd);
        return posEnd;
}
