/*
 * AbiCollab - Code to enable the modification of remote documents.
 * Copyright (C) 2006 by Marc Maurer <uwog@uwog.net>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA
 * 02111-1307, USA.
 */
 
#ifndef ABICOLLAB_PACKET_H
#define ABICOLLAB_PACKET_H

#include <libxml/xmlreader.h>

#include "ut_types.h"
#include "ut_string_class.h"
#include "px_ChangeRecord.h"
#include "ut_stack.h"

class Buddy;

enum PacketType
{
	PT_Session = 0,
	PT_Event,
	PT_Handler
};
	
enum PacketSessionType
{
	PACKET_CHANGERECORD,
	PACKET_SIGNAL,
	PACKET_GLOB,
	PACKET_REVERT,
	PACKET_REVERT_ACK
};

enum PacketEventType
{
	PTE_AccountAddBuddyRequest,

	PTE_StartSession,
	PTE_JoinSession,
	PTE_DisjoinSession,
	PTE_CloseSession,
};

struct RawPacket
{
	Buddy* buddy; // TODO: free the buddy somewhere, or make it not a pointer (please do the latter) - MARCM
	UT_UTF8String packet;
};

class PX_ChangeRecord;
class SessionPacket;

class ABI_EXPORT SessionPacketFactory
{
public:
	SessionPacketFactory(xmlNode* sessionNode);
	
	SessionPacket *		getNextPacket(void);
	bool				getChangeRecordType(const UT_UTF8String& sType, 
								PX_ChangeRecord::PXType & crType);
	UT_uint32			getSignalType(const UT_UTF8String& sType);

private:
	SessionPacket*		_getNextPacket(xmlNode* pPacketNode, const UT_UTF8String& sSessionId);
	SessionPacket*		_getNextSessionPacket(xmlNode* pSessionNode);
	
	xmlNode*			m_pCurNode;
};

/*************************************************************
 * Packets                                                   *
 *************************************************************/

class ABI_EXPORT Packet
{
public:
	Packet();
	virtual ~Packet();

	virtual PacketType				getType() const = 0;

	bool							fillPTAttributes(xmlNode * node);
	bool							fillPropsFromString(UT_UTF8String& sProps); 
	char *							getAttribute(char * szName) const;
		
	// NOTE: is an UT_UTF8String the best format for this?
	virtual const UT_UTF8String		serialize() = 0;

protected:
	gchar**						m_szAtts;
	gchar**						m_szProps;
	void							_freeProps(); 
	void							_freeAtts();
};


/*************************************************************
 * SessionPackets                                            *
 *************************************************************/

class ABI_EXPORT SessionPacket : public Packet
{
public:
	SessionPacket(const UT_UTF8String& sSessionId);
	virtual ~SessionPacket();
	
	virtual PacketType getType() const
		{ return PT_Session; }

	virtual PacketSessionType getSessionType() const = 0;

	const UT_UTF8String& getSessionId()
		{ return m_sSessionId; }

	void setSessionId(const UT_UTF8String& sessionId)
		{ m_sSessionId = sessionId; }

	// FIXME: This function should be implemented by the specific subclasses
	const UT_UTF8String				serialize()
		{ return "TODO: IMPLEMENT ME"; }
		
private:
	UT_UTF8String			m_sSessionId;	
};

class ABI_EXPORT ChangeRecordSessionPacket : public SessionPacket
{
public:
	ChangeRecordSessionPacket(
			const UT_UTF8String& sSessionId,
			PX_ChangeRecord::PXType cType, 
			const UT_UTF8String& sDocUUID, 
			int iRev, 
			int iRemoteRev, 
			int iPos,
			UT_Byte iGLOB,
			PacketSessionType crType);
	
	virtual ~ChangeRecordSessionPacket(void);

	virtual PacketSessionType getSessionType() const
		{ return m_crType; }

	void setSpanProps(size_t iLength, const char* value);

	PX_ChangeRecord::PXType getPXType() const
		{ return m_cType; }

	const UT_UTF8String& getDocUUID() const
		{ return m_sDocUUID; }
	
	UT_sint32 getRev() const
		{ return m_iRev; }

	UT_sint32 getRemoteRev(void) const
		{ return  m_iRemoteRev; }

	UT_sint32 getPos() const
		{ return m_iPos; }

	UT_Byte getGLOBType() const
		{ return m_iGLOBType;}

	const UT_UTF8String& getFrag() const
		{ return m_sFrag; }	
		
	UT_sint32 getLength() const
		{ return m_iLength; }

	UT_sint32 getAdjust() const
		{ return m_iAdjust; }

	void setAdjust(UT_sint32 iAdjust)
		{ m_iAdjust = iAdjust; }
		
	gchar** getProps() const
		{ return m_szProps; }
		
	gchar** getAtts() const
		{ return m_szAtts; }
		
	const UT_UTF8String& getValue() const
		{ return m_sValue; }

	void setFrag(const xmlChar * szFrag)
		{ m_sFrag = reinterpret_cast<const char *>(szFrag); }

private:
	PX_ChangeRecord::PXType		m_cType;
	UT_UTF8String				m_sDocUUID;
	UT_sint32					m_iRev;
	UT_sint32					m_iRemoteRev;
	UT_sint32					m_iPos;
	
	UT_UTF8String				m_sFrag;
	UT_sint32					m_iLength;
	UT_sint32					m_iAdjust;

	UT_UTF8String				m_sValue;
	UT_Byte						m_iGLOBType;

	PacketSessionType			m_crType;
};

class ABI_EXPORT SignalSessionPacket : public SessionPacket
{
public:
	SignalSessionPacket(const UT_UTF8String& sSessionId, UT_uint32 iSignal);

	virtual const UT_UTF8String* serialize() const;

	virtual PacketSessionType getSessionType() const
		{ return PACKET_SIGNAL; }
	
	UT_uint32 getSignalType() const
		{ return m_iSignal; }
		
private:
	UT_uint32	m_iSignal;
};

class ABI_EXPORT GlobSessionPacket : public SessionPacket
{
public:
	GlobSessionPacket(const UT_UTF8String& sessionId);
	virtual ~GlobSessionPacket();

	virtual PacketSessionType getSessionType() const
		{ return PACKET_GLOB; }

	void addPacket(SessionPacket* pPacket)
		{ m_pPackets.push_back(pPacket); }

	const UT_GenericVector<SessionPacket*>& getPackets() const
		{ return m_pPackets; }
		
private:
	UT_GenericVector<SessionPacket*>		m_pPackets;
};

class ABI_EXPORT RevertSessionPacket : public SessionPacket
{
public:
	RevertSessionPacket(const UT_UTF8String& sessionId, UT_sint32 iRev);
	virtual ~RevertSessionPacket();

	virtual const UT_UTF8String* serialize() const;

	virtual PacketSessionType getSessionType() const
		{ return PACKET_REVERT; }

	UT_sint32			getRev() const
		{ return m_iRev; }

private:
	UT_sint32			m_iRev;
};

class ABI_EXPORT RevertAckSessionPacket : public SessionPacket
{
public:
	RevertAckSessionPacket(const UT_UTF8String& sessionId, UT_sint32 iRev);
	virtual ~RevertAckSessionPacket();

	virtual const UT_UTF8String* serialize() const;

	virtual PacketSessionType getSessionType() const
		{ return PACKET_REVERT_ACK; }

	UT_sint32			getRev() const
		{ return m_iRev; }

private:
	UT_sint32			m_iRev;
};

/*************************************************************
 * HandlerPackets                                            *
 *************************************************************/

// ...

#endif /* ABICOLLAB_PACKET_H */
