/* Copyright (C) 2006,2007 by Marc Maurer <uwog@uwog.net>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA
 * 02111-1307, USA.
 */

#include "ut_path.h"

#include <xp/AbiCollab_Plugin.h>
#include <xp/AbiCollab.h>
#include <xp/AbiCollab_Export.h>
#include <xp/AbiCollabSessionManager.h>
#include "AccountEvent.h"
#include "AccountHandler.h"

const string AccountHandler::getProperty(const string& key)
{ 
	PropertyMap::iterator pos = m_properties.find(key);
	if (pos != m_properties.end())
		return pos->second;
	else
		return "";
}

bool AccountHandler::autoConnect()
{
	const std::string autoconnect = getProperty("autoconnect");
	return strcmp(autoconnect.c_str(), "true") == 0;
}

bool AccountHandler::operator==(AccountHandler & rhHandler) {
	// Backends should override this if they need specific properties blacklisted beyond just autoconnect
	PropertyMap::iterator iter = m_properties.begin();
	PropertyMap::iterator otherMapIter;
	PropertyMap::iterator end = m_properties.end();
	
	// If different number of properties, then they can't be the same
	// Assumes that no items on the blacklist might be missing - this could be a false assumption.
	bool returnval=(m_properties.size() == rhHandler.m_properties.size());
	while (returnval && iter != end)
	{
		// Check to see if property is on the blacklist
		// TODO: replace this with a vector if we ever need more than one on the blacklist.
		if ((iter->first) != "autoconnect")
		{
			// not on the blacklist
			if ((otherMapIter = rhHandler.m_properties.find((iter->first))) != rhHandler.m_properties.end())
			{
				// if the other property map has this property
				returnval = ((iter->second) == (otherMapIter->second));
			}
		}
		iter++;
	}
	return returnval;
}

/*!
 Send a collaboration packet to all registered buddies
 */ 
bool AccountHandler::send(AbiCollab* pSession, const UT_UTF8String& packet, PacketType packetType)
{
	// NOTE: pSession may be NULL for certain packets
	UT_DEBUGMSG(("AccountHandler::send()\n"));

	UT_UTF8String sDecoratedPacket;
	if (_decoratePacket(pSession, packet, packetType, sDecoratedPacket))
	{
//		if (packetType == PT_Session)
//		{
//			UT_DEBUGMSG(("packet send delay...\n"));
//			usleep(5*1000*1000);
//			UT_DEBUGMSG(("sending...\n"));
//		}
		return _send(sDecoratedPacket);
	}
	return false;
}

/*!
  Send a collaboration packet to a buddy
 */	
bool AccountHandler::send(AbiCollab* pSession, const UT_UTF8String& packet, PacketType packetType, const Buddy& buddy)
{
	// NOTE: pSession may be NULL for certain packets
	UT_DEBUGMSG(("AccountHandler::send()\n"));

	UT_UTF8String sDecoratedPacket;
	if (_decoratePacket(pSession, packet, packetType, sDecoratedPacket))
	{
//		if (packetType == PT_Session)
//		{
//			UT_DEBUGMSG(("packet send delay...\n"));
//			usleep(5*1000*1000);
//			UT_DEBUGMSG(("sending...\n"));
//		}
		UT_DEBUGMSG(("Sending decorated packet: %s\n", sDecoratedPacket.utf8_str()));
		return _send(sDecoratedPacket, buddy);
	}
	return false;
}

void AccountHandler::addBuddy(Buddy* buddy)
{
	m_vecBuddies.push_back(buddy);
	
	// signal all listeners we have a new buddy
	AccountAddBuddyEvent event;
	// TODO: fill the event
	AbiCollabSessionManager::getManager()->signal(event);
}

Buddy* AccountHandler::getBuddy(const UT_UTF8String& name)
{
	for (UT_sint32 i = 0; i < m_vecBuddies.getItemCount(); i++)
	{
		Buddy* pBuddy = m_vecBuddies.getNthItem(i);
		if (pBuddy->getName() == name)
			return pBuddy;
	}		
	UT_DEBUGMSG(("Getting buddy (%s) failed\n", name.utf8_str()));
	return 0;
}
		
void AccountHandler::getSessionsAsync()
{
	for (UT_sint32 i = 0; i < m_vecBuddies.getItemCount(); i++)
	{
		const Buddy* pBuddy = m_vecBuddies.getNthItem(i);
		getSessionsAsync(*pBuddy);
	}
}

void AccountHandler::getSessionsAsync(const Buddy& buddy)
{
	UT_UTF8String sDocRequest = "<GetSessionsRequest/>";
	send(NULL, sDocRequest, PT_Handler, buddy);
}

void AccountHandler::joinSessionAsync(const Buddy& buddy, DocHandle& docHandle)
{
	UT_UTF8String sDocRequest = "<JoinSessionRequest>";
	sDocRequest += "<sessionId>";
	sDocRequest += docHandle.getSessionId();
	sDocRequest += "</sessionId>";
	sDocRequest += "</JoinSessionRequest>";
	send(NULL, sDocRequest, PT_Handler, buddy);
}

void AccountHandler::signal(const Event& event, const Buddy* pSource)
{
	UT_DEBUGMSG(("AccountHandler::signal()\n"));

	// broadcast this event over our network (if applicable for each message type)
	EventPacket* pPacket = event.constructPacket();
	if (pPacket)
	{
		UT_GenericVector<Buddy*> vRecipients = 
			(event.isBroadcast() ? getBuddies() : event.getRecipients());
		
		for (UT_sint32 i = 0; i < vRecipients.getItemCount(); i++)
		{
			Buddy* pRecipient = vRecipients.getNthItem(i);
			if (pRecipient)
			{
				if (!pSource || 
					(pSource && pRecipient->getName() != pSource->getName())
					)
				{
					send(0, pPacket->serialize(), pPacket->getType(), *pRecipient);
				}
				else
				{
					// the event came originated at this buddy, so make sure not to send it
					// back to him, as it would result in a broadcast storm and
					// kill the network really fast
				}
			}
		}
	}
}

void AccountHandler::handleMessage(RawPacket* pRp)
{
	UT_return_if_fail(pRp);
	UT_return_if_fail(pRp->buddy);
			
	// TODO: most likely, this packet parsing code can be moved to AccountHandler itself

	// First capture any request that is destined for the account handler itself
	xmlDocPtr doc = xmlReadMemory (pRp->packet.utf8_str(), pRp->packet.length(), 0, "UTF-8", 0);
	UT_return_if_fail(doc);
	
	xmlNode* protocolNode = xmlDocGetRootElement(doc);
	UT_return_if_fail(protocolNode);
	
	// we only expect 1 root element - do we really? - MARCM
	UT_ASSERT(protocolNode->next == 0);
		
	if (strcmp(reinterpret_cast<const char*>(protocolNode->name), "abicollab") == 0) 
	{
		UT_UTF8String sPacketProtocolVersion = reinterpret_cast<char *>(xmlGetProp(protocolNode, reinterpret_cast<const xmlChar*>("protocol")));
		if (sPacketProtocolVersion == ABICOLLAB_PROTOCOL_VERSION)
		{
			xmlNode* packetTypeNode = protocolNode->children;
			if (packetTypeNode)
			{
				if (strcmp(reinterpret_cast<const char*>(packetTypeNode->name), "session") == 0)
				{
					// this is a session packet
					AbiCollabSessionManager* pManager = AbiCollabSessionManager::getManager();
					pManager->processSessionPacket(*this, packetTypeNode, pRp->buddy->getName());
				}
				else if (strcmp(reinterpret_cast<const char*>(packetTypeNode->name), "event") == 0)
				{
					// this is an event packet
					AbiCollabSessionManager* pManager = AbiCollabSessionManager::getManager();
					pManager->processEventPacket(*this, packetTypeNode, pRp->buddy->getName());
				}
				else if (strcmp(reinterpret_cast<const char*>(packetTypeNode->name), "handler") == 0)
				{
					// yay, this is a packet for us!
					xmlNode* handlerNode = packetTypeNode->children;
					if (handlerNode)
					{
						if(strcmp(reinterpret_cast<const char*>(handlerNode->name), "GetSessionsRequest") == 0)
						{
							_handleGetSessionsRequest(handlerNode, pRp);
						}
						else if (strcmp(reinterpret_cast<const char*>(handlerNode->name), "GetSessionsResponse") == 0)
						{
							_handleGetSessionsResponse(handlerNode, pRp);
						}
						else if(strcmp(reinterpret_cast<const char*>(handlerNode->name), "JoinSessionRequest") == 0)
						{
							_handleJoinSessionRequest(handlerNode, pRp);
						}
						else if(strcmp(reinterpret_cast<const char*>(handlerNode->name), "JoinSessionResponse") == 0)
						{
							_handleJoinSessionResponse(handlerNode, pRp);
						}
					}
				}
				else
				{
					// huh?!
					UT_ASSERT(false);
				}
			}
		}
		else
		{
			// TODO: protocol version mismatch!
		}
	}

	DELETEP(pRp);
	xmlFreeDoc(doc);
}

bool AccountHandler::_decoratePacket(const AbiCollab* pSession, const UT_UTF8String& packet, const PacketType packetType, UT_UTF8String& sDecoratedPacket)
{
	// NOTE: pSession may be NULL for certain packets
	UT_DEBUGMSG(("AccountHandler::_decoratePacket()\n"));

	// TODO: merge this code with ::send(pSession, packet)
	if (packetType == PT_Session && pSession == 0)
	{
		UT_ASSERT(UT_SHOULD_NOT_HAPPEN);
		return false;
	}
	
	sDecoratedPacket += "<abicollab protocol=\"";
	sDecoratedPacket += ABICOLLAB_PROTOCOL_VERSION;
	sDecoratedPacket += "\">";
	
	switch (packetType)
	{
		case PT_Session:
			sDecoratedPacket += "<session id=\"";
			sDecoratedPacket += pSession->getSessionId();
			sDecoratedPacket += "\">";
			sDecoratedPacket += packet;
			sDecoratedPacket += "</session>";
			break;
		case PT_Event:
			sDecoratedPacket += "<event>";
			sDecoratedPacket += packet;
			sDecoratedPacket += "</event>";
			break;
		case PT_Handler:
			sDecoratedPacket += "<handler>";
			sDecoratedPacket += packet;
			sDecoratedPacket += "</handler>";
			break;
		default:
			UT_ASSERT(UT_SHOULD_NOT_HAPPEN);
			break;
	}
	sDecoratedPacket += "</abicollab>";
	
	return true;
}

/* 
 * packet handling functions 
 */

void AccountHandler::_handleGetSessionsRequest(xmlNode* parent, RawPacket* pRp)
{
	UT_DEBUGMSG(("AccountHandler::_handleGetSessionsRequest()\n"));
	UT_return_if_fail(pRp);

	// A remote user has requested our open and connected documents
	// Return an xml-ized list of them.
	UT_UTF8String sDocResponse = "<GetSessionsResponse>";
	AbiCollabSessionManager* pManager = AbiCollabSessionManager::getManager();
	const UT_GenericVector<AbiCollab *> sessions = pManager->getSessions();
	for (UT_sint32 i = 0; i < sessions.getItemCount(); i++)
	{
		AbiCollab* pSession = sessions.getNthItem(i);
		if (pSession && pSession->isLocallyControlled())
		{
			const PD_Document * pDoc = pSession->getDocument();
			if (pDoc)
			{
				sDocResponse += "<session name=\"";
				// Let's use the document name for the session name
				// TODO: check the metadata first for a nicer document name
				if (pDoc->getFilename())
				{
					UT_UTF8String sDocumentBaseName(UT_go_basename_from_uri(pDoc->getFilename()));
					sDocResponse += sDocumentBaseName;
				}			
				sDocResponse += "\" id=\"";
				sDocResponse += pSession->getSessionId();
				sDocResponse += "\"/>";
			}
			else
				UT_ASSERT(UT_SHOULD_NOT_HAPPEN);
		}
	}
	sDocResponse +="</GetSessionsResponse>";
	send(NULL, sDocResponse, PT_Handler, *pRp->buddy);
}

void AccountHandler::_handleGetSessionsResponse(xmlNode* parent, RawPacket* pRp)
{
	UT_DEBUGMSG(("AccountHandler::_handleGetSessionsResponse()\n"));
	UT_return_if_fail(parent);
	UT_return_if_fail(pRp);
	
	AbiCollabSessionManager* pManager = AbiCollabSessionManager::getManager();
	UT_return_if_fail(pManager);
	
	Buddy* pBuddy = getBuddy(pRp->buddy->getName());
	if (pBuddy != NULL)
	{
		UT_GenericVector<DocHandle*> vDocHandles;
		for (xmlNode* document_node = parent->children; document_node; document_node = document_node->next)
		{
			if (document_node->type == XML_ELEMENT_NODE)
			{
				if (strcmp(reinterpret_cast<const char*>(document_node->name), "session") == 0)
				{
					UT_UTF8String sSessionId = reinterpret_cast<char *>(xmlGetProp(document_node, reinterpret_cast<const xmlChar*>("id")));
					UT_DEBUGMSG(("Got a session with id %s\n", sSessionId.utf8_str()));
					UT_UTF8String sSessionName = reinterpret_cast<char *>(xmlGetProp(document_node, reinterpret_cast<const xmlChar*>("name"))); 
					
					DocHandle* pDocHandle = new DocHandle(sSessionId, sSessionName);
					vDocHandles.addItem(pDocHandle);
				}
			}
		}
		pManager->setDocumentHandles(*pBuddy, vDocHandles);
	}
}

void AccountHandler::_handleJoinSessionRequest(xmlNode* parent, RawPacket* pRp)
{
	UT_DEBUGMSG(("AccountHandler::_handleJoinSessionRequest()\n"));
	UT_return_if_fail(pRp);
	
	AbiCollabSessionManager* pManager = AbiCollabSessionManager::getManager();
	UT_return_if_fail(pManager);

	UT_UTF8String sSessionId;
	// parse all document properties in this request
	for (xmlNode* prop = parent->children; prop; prop = prop->next)
	{
		if (prop->type == XML_ELEMENT_NODE)
		{
			if (strcmp(reinterpret_cast<const char*>(prop->name), "sessionId") == 0)
			{
				sSessionId = reinterpret_cast<const char*>(xmlNodeGetContent(prop));
				UT_DEBUGMSG(("Incoming JoinSessionRequest for session id |%s|\n", sSessionId.utf8_str()));
			}
		}
	}

	AbiCollab* pSession = pManager->getSessionFromSessionId(sSessionId);
	UT_return_if_fail(pSession);

	ABI_Collab_Export* pExport = pSession->getExport();
	UT_return_if_fail(pExport);
	
	const UT_GenericVector<ChangeAdjust *>* pExpAdjusts = pExport->getAdjusts();
	UT_return_if_fail(pExpAdjusts);

	// TODO: ask the user to authorize this request
	bool bAuthorized = true;
	if (bAuthorized)
	{
		AbiCollabSessionManager* pManager = AbiCollabSessionManager::getManager();
		AbiCollab* pAbiCollab = pManager->getSessionFromSessionId(sSessionId);
		if (pAbiCollab)
		{
			PD_Document* pDoc = pAbiCollab->getDocument();
		
			UT_UTF8String sBase64GzDocument;
			if (AbiCollabSessionManager::serializeDocument(pDoc, sBase64GzDocument) == UT_OK)
			{
				UT_UTF8String sDocResponse = "<JoinSessionResponse>\n";
				sDocResponse += "<document id=\"";
				sDocResponse += pDoc->getDocUUIDString();
				sDocResponse += "\" sessionId=\"";
				sDocResponse += sSessionId.utf8_str();
				sDocResponse += "\" name=\"";
				// TODO: check the metadata first for a nicer document name
				if (pDoc->getFilename())
				{
					UT_UTF8String sDocumentBaseName(UT_go_basename_from_uri(pDoc->getFilename()));
					sDocResponse += sDocumentBaseName;
				}
				sDocResponse += "\" rev=\"";
				UT_sint32 iRev = (pExpAdjusts->getItemCount() > 0 ? pExpAdjusts->getNthItem(pExpAdjusts->getItemCount()-1)->m_iCRNumber : 0);
				sDocResponse += UT_UTF8String_sprintf("%d", iRev);
				sDocResponse +=	"\">";
				sDocResponse += sBase64GzDocument;
				sDocResponse += "</document>\n";
				sDocResponse +="</JoinSessionResponse>";
				send(NULL, sDocResponse, PT_Handler, *pRp->buddy);
			}
			
			// NOTE: this is kinda wacky atm, but for now, when someone has requested a document,
			// we'll consider him joined to the session as well
			Buddy* pCollaborator = pRp->buddy; // ICK ICK, please please implement a proper clone() function
			
			if (pCollaborator)
			{
				// check if we already know this buddy
				Buddy* existing = getBuddy(pCollaborator->getName());
				if (!existing)
				{
					// we don't know this buddy yet; add this one as a volatile buddy
					pCollaborator->setVolatile(true);
					addBuddy(pCollaborator);
				}
			
				// add this buddy to the collaboration session
				pAbiCollab->addCollaborator(pCollaborator);
				pRp->buddy = 0; // ICK ICK, please please implement a proper clone() function
			}
			else
				UT_ASSERT(UT_SHOULD_NOT_HAPPEN);
		}
		else
			UT_DEBUGMSG(("Requesting a session with an unknown id (%s)!\n", sSessionId.utf8_str()));
	}
}

void AccountHandler::_handleJoinSessionResponse(xmlNode* parent, RawPacket* pRp)
{
	UT_DEBUGMSG(("AccountHandler::_handleJoinSessionResponse()\n"));
	UT_return_if_fail(pRp);

	// we only expect 1 document per response
	UT_ASSERT(parent->next == 0);
	
	// get the document's contents
	UT_UTF8String sDocumentId;
	UT_UTF8String sSessionId;
	UT_UTF8String sDocumentName;
	UT_UTF8String sDocumentContent;
	UT_sint32 iRev;
	for (xmlNode* prop = parent->children; prop; prop = prop->next)	
	{
		if (prop->type == XML_ELEMENT_NODE)
		{
			if (strcmp(reinterpret_cast<const char*>(prop->name), "document") == 0)
			{
				// get the document id
				char* sz = (char *)xmlGetProp(prop, (const xmlChar *)"id");
				sDocumentId = sz;
				FREEP(sz);

				// get the session id
				sz = (char *)xmlGetProp(prop, (const xmlChar *)"sessionId");
				sSessionId = sz;
				FREEP(sz);
				
				// get the document name
				sz = (char *)xmlGetProp(prop, (const xmlChar *)"name");
				sDocumentName = sz;
				FREEP(sz);

				// get the document revision
				sz = (char *)xmlGetProp(prop, (const xmlChar *)"rev");
				UT_UTF8String sRev = sz;
				FREEP(sz);
				if(sRev.size() > 0)
					iRev = atoi(sRev.utf8_str());

				// get the document id
				// TODO: do we need this? if so, then implement this
				
				// get the document contents
				sDocumentContent = reinterpret_cast<const char*>(xmlNodeGetContent(prop));
			}
		}
	}
	
	Buddy* pCollaborator = pRp->buddy;			
	if (pCollaborator)
	{
		AbiCollabSessionManager* pManager = AbiCollabSessionManager::getManager();
		if (pManager)
		{
			PD_Document* pDoc = 0;
			if (AbiCollabSessionManager::deserializeDocument(&pDoc, sDocumentContent) == UT_OK)
			{
				if (pDoc)
				{
					// NOTE: we could adopt the same document name here, but i'd
					// rather not at the moment - MARCM
					pDoc->forceDirty();
					pManager->joinSession(sSessionId, pDoc, sDocumentId, iRev, pCollaborator);
				}
				else
					UT_DEBUGMSG(("XMPPAccountHandler::_handleJoinSessionResponse() - deserializing document failed!\n"));
			}
		}
	}
}

#ifdef WIN32
BOOL AccountHandler::_onCommand(HWND hWnd, WPARAM wParam, LPARAM lParam)
{
	return 1; // not handled by default
}
#endif